/*
** Source Lines of Code Counter         (c) Copyright IBM Corp., 1987, 1992
**                                      For Internal Use Only
**    by: Jeffrey W. Hamilton  (JEFFH at WMAVM7)
*/
#include "global.h"

extern int debug;
extern char *profile_fn;

static re_prog_t *find_ext, *find_key, *find_range, *find_indicator;
static int current_line;

void build_profile(FILE *profile)
{
   description *temp_descr;

   /*
   ** Build a regular expression program for the major profile components
   */

   /*
   ** An extension is a set of non-blank characters terminated by a colon.
   ** The set of characters will be stored in register 1.
   */
   find_ext = re_compile("[ \t]*([^ \t:]*)[ \t]*:");
   if (find_ext == NULL) {
      fprintf(stderr,"Internal error: find_ext is NULL\n");
      fprintf(stderr,"re_compile returns %s\n",re_error_string);
      exit(2);
   }
   /*
   ** A keyword is a set of non-blank characters terminated by an equal sign.
   ** The set of characters will be stored in register 1.
   */
   find_key = re_compile("[ \t]*([^ \t=]*)[ \t]*=");
   if (find_key == NULL) {
      fprintf(stderr,"Internal error: find_key is NULL\n");
      fprintf(stderr,"re_compile returns %s\n",re_error_string);
      exit(2);
   }
   /*
   ** A range is a start indicator bracketed by <> followed by ... or .n.
   ** followed by an end indicator bracketed by <>.
   ** Register 1: Placement code for the start indicator
   ** Register 2: The start indicator
   ** Register 3: Either . on n to indicate nestability
   ** Register 4: Placement code for the end indicator
   ** Register 5: The end indicator
   */
   find_range = re_compile("[ \t]*([Ss]|[Ll]|[Ff][0-9]+|[Cc][0-9]+|[^<]*)<([^>]*)>[ \t]*\\.([.n])\\.[ \t]*([Ss]|[Ll]|[Ff][0-9]+|[Cc][0-9]+|[^>]*)<([^>]*)>");
   if (find_range == NULL) {
      fprintf(stderr,"Internal error: find_range is NULL\n");
      fprintf(stderr,"re_compile returns %s\n",re_error_string);
      exit(2);
   }
   /*
   ** An indicator is bracketed by <>. An optional placement code is placed in
   ** register 1. The indicator will be placed in register 2.
   ** The definition has a tolerable flaw: the user cannot use > in the
   ** indicator, even if it is preceded with a backslash.
   */
   find_indicator = re_compile("[ \t]*([Ss]|[Ll]|[Ff][0-9]+|[Cc][0-9]+|[^<]*)<([^>]*)>");
   if (find_range == NULL) {
      fprintf(stderr,"Internal error: find_indicator is NULL\n");
      fprintf(stderr,"re_compile returns %s\n",re_error_string);
      exit(2);
   }

   /*
   ** Read each description and build an entry for the file extension
   */
   current_line = 0;
   while (!feof(profile)) {
      temp_descr = parse_extension(profile);
      if (temp_descr == NULL) break;   /* End of file reached early */
      temp_descr->next = descr;
      descr = temp_descr;
   }

   /* Release space we no longer need */
   re_destroy(find_ext);
   re_destroy(find_key);
   re_destroy(find_indicator);
}

/*
** Process a single description of a file extension. A description starts
** with an file extension and ends with a blank line or the end of the file.
*/
description *parse_extension(FILE *profile)
{
   char *cpos;                    /* Current position in the line */
   char string[40];
   description *descr;
   alt_mode *mode, *next_mode;
   sym_pair *pair;
   int offset;

   do {
      current_line += 1;
      cpos = fgets(buffer,512,profile);
      if (cpos == NULL) return(NULL); /* Abort on error */
      strip_blanks(cpos);
   } while ((*cpos == '-') || (*cpos == '\0'));

   /* First parameter must be a file extension */
   if ((offset = re_search(find_ext,cpos)) == -1) {
      /* invalid file extension */
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Missing or incorrectly written file extension\n");
      exit(2);
   }
   cpos += offset;
   if (re_read_register(find_ext,string,40,1) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Incorrectly written file extension\n");
      exit(2);
   }
   if (strlen(string) > EXTENSION_LIMIT) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"File extension exceeds %d characters\n", EXTENSION_LIMIT);
      exit(2);
   }

   /* Create a node to hold description */
   descr = malloc(sizeof(description));
   check_space(descr);
   if ((descr->ext = re_compile(string)) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"%s\n",re_error_string);
      exit(2);
   }
   mode = malloc(sizeof(alt_mode));
   check_space(mode);
   descr->mode = mode;
   mode->next = NULL;
   mode->alternate = NULL;
   mode->comment = NULL;
   mode->ignore = NULL;
   mode->escape = NULL;
   mode->line = NULL;
   mode->view = NULL;
   mode->cont_next = NULL;
   mode->cont_prev = NULL;

   /* Fill in the node with the appropriate information */
   /* a blank line indicates the end of a description */
   strip_blanks(cpos);
   if (cpos == '\0') {
      current_line += 1;
      if ((cpos = fgets(buffer,512,profile)) == NULL) return(descr);
      strip_blanks(cpos);
   }
   while (*cpos != '\0') {
      /* Process the next keyword in the profile */
      if ((offset = re_search(find_key,cpos)) == -1) {
         /* invalid keyword */
         fprintf(stderr,"Line %d in language description file %s\n",
                 current_line, profile_fn);
         fprintf(stderr,"Missing or incorrectly written description keyword\n");
         exit(2);
      }
      cpos += offset;
      if (re_read_register(find_key,string,40,1) == NULL) {
         fprintf(stderr,"Line %d in language description file %s\n",
                 current_line, profile_fn);
         fprintf(stderr,"Incorrectly written description keyword\n");
         exit(2);
      }
      /* Comment Field */
      if (strcmpi(string,"COMMENTS") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->comment;
            mode->comment = pair;
         }
      /* Ignore Field */
      } else if (strcmpi(string,"IGNORE") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->ignore;
            mode->ignore = pair;
         }
      /* Alternative Mode Field */
      } else if (strcmpi(string,"ALTERNATE") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->alternate;
            mode->alternate = pair;
         }
         next_mode = malloc(sizeof(alt_mode));
         check_space(next_mode);
         mode->next = next_mode;
         mode = next_mode;
         mode->next = NULL;
         mode->alternate = NULL;
         mode->comment = NULL;
         mode->ignore = NULL;
         mode->escape = NULL;
         mode->line = NULL;
         mode->view = NULL;
         mode->cont_next = NULL;
         mode->cont_prev = NULL;
      /* Line Field */
      } else if (strcmpi(string,"LINE") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->line;
            mode->line = pair;
         }
      /* Escape character */
      } else if (strcmpi(string,"ESCAPE") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->escape;
            mode->escape = pair;
         }
      /* Limited line view markers */
      } else if (strcmpi(string,"VIEW") == 0) {
         while ((pair = parse_pair(&cpos)) != NULL) {
            pair->next = mode->view;
            mode->view = pair;
         }
      /* Continuation Field */
      } else if (strcmpi(string,"CONTINUE_ON_NEXT") == 0) {
         strip_blanks(cpos);
         while ((*cpos != '\0') && (*cpos != '\n')) {
            pair = malloc(sizeof(sym_pair));
            check_space(pair);
            parse_marker(&cpos,&pair->start);
            pair->next = mode->cont_next;
            mode->cont_next = pair;
            strip_blanks(cpos);
         }
      } else if (strcmpi(string,"CONTINUE_FROM_LAST") == 0) {
         strip_blanks(cpos);
         while ((*cpos != '\0') && (*cpos != '\n')) {
            pair = malloc(sizeof(sym_pair));
            check_space(pair);
            parse_marker(&cpos,&pair->start);
            pair->next = mode->cont_prev;
            mode->cont_prev = pair;
            strip_blanks(cpos);
         }
      } else {
         fprintf(stderr,"Line %d in language description file %s\n",
                 current_line, profile_fn);
         fprintf(stderr,"%s is an unknown written description keyword\n",string);
         exit(2);
      }
      current_line += 1;
      if ((cpos = fgets(buffer,512,profile)) == NULL) return(descr);
      strip_blanks(cpos);
   }
   return(descr);
}

/*
** Isolate a pair of starting and ending markers. NULL is returned if the
** end of a line is reached.
*/
sym_pair *parse_pair(char **pos)
{
   sym_marker start,end;
   sym_pair *pair;
   char string[100];
   int offset;
   int nestable;

   /* Check for end of the line */
   strip_blanks(*pos);
   if ((**pos == '\n') || (**pos == '\0')) return NULL;

   /* Parse the entire range */
   if ((offset = re_search(find_range,*pos)) == -1) {
      /* invalid range */
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Missing or incorrectly written indicator range\n");
      exit(2);
   }

   *pos += offset;

   /* Pick up the start position marker */
   if (re_read_register(find_range,string,10,1) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Incorrectly written starting position\n");
      exit(2);
   }
   parse_position(string, &start);

   /* Pick up the start expression marker */
   if (re_read_register(find_range,string,100,2) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Starting indicator must be less than 100 characters\n");
      exit(2);
   }
   if ((start.expression = re_compile(string)) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"%s\n",re_error_string);
      exit(2);
   }

   /* Pick up the nestable flag */
   if (re_read_register(find_range,string,2,3) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Incorrectly written range separator\n");
      exit(2);
   }
   if (toupper(string[0]) == '.') {
      nestable = 0;
   } else if (toupper(string[0]) == 'N') {
      nestable = 1;
   } else {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Range separator is not ... or .n.\n");
      exit(2);
   }

   /* Pick up the end position marker */
   if (re_read_register(find_range,string,10,4) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Incorrectly written ending position\n");
      exit(2);
   }
   parse_position(string, &end);

   /* Pick up the start expression marker */
   if (re_read_register(find_range,string,100,5) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Ending indicator must be less than 100 characters\n");
      exit(2);
   }
   if ((end.expression = re_compile(string)) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"%s\n",re_error_string);
      exit(2);
   }

   pair = malloc(sizeof(sym_pair));
   check_space(pair);
   memcpy(&pair->start,&start,sizeof(start));
   memcpy(&pair->end,&end,sizeof(end));
   pair->nestable = nestable;
   return(pair);
}

/*
** Parse a marker's position flag
*/
void parse_position(char *string, sym_marker *marker)
{
   switch (toupper(string[0])) {
   case '\0':
      /* No special position requested */
      marker->type = FREE_FORM;
      break;
   case 'S':
      /* At the beginning of a line */
      marker->type = AT_START;
      break;
   case 'F':
      /* In field n */
      marker->type = FIELD;
      marker->position = (int) strtol(&string[1], NULL, 10);
      break;
   case 'C':
      /* In column n */
      marker->type = COLUMN;
      marker->position = (int) strtol(&string[1], NULL, 10);
      break;
   case 'L':
      marker->type = AT_LINE_START;
      break;
   default:
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"%c is not a recognized position for an indicator\n", string[0]);
      exit(2);
   }
}

/*
** Isolate a single marker.
*/
void parse_marker(char **pos, sym_marker *marker)
{
   int offset;
   char string[100];

   /* Check for end of the line */
   strip_blanks(*pos);
   if ((**pos == '\n') || (**pos == '\0')) {
      marker->expression = NULL;
      return;
   }

   /* Parse the marker */
   if ((offset = re_search(find_indicator,*pos)) == -1) {
      /* invalid range */
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Missing or incorrectly written indicator\n");
      exit(2);
   }

   *pos += offset;

   /* Pick up the position marker */
   if (re_read_register(find_indicator,string,10,1) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Incorrectly written position\n");
      exit(2);
   }
   parse_position(string, marker);

   /* Pick up the expression */
   if (re_read_register(find_indicator,string,100,2) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"Indicator must be less than 100 characters\n");
      exit(2);
   }
   if ((marker->expression = re_compile(string)) == NULL) {
      fprintf(stderr,"Line %d in language description file %s\n",
              current_line, profile_fn);
      fprintf(stderr,"%s\n",re_error_string);
      exit(2);
   }
}
